﻿using System;
using System.Net.Http.Headers;
using System.Text;
using System.Net.Http;
using Newtonsoft.Json.Linq;
using Newtonsoft.Json;
using System.Collections.Generic;
using Microsoft.Extensions.Configuration;
using System.Threading.Tasks;

namespace CSHttpClientSample
{
    static class Program
    {
        static IConfiguration Configuration;
        static string AccessToken;

        static async Task Main(string[] args)
        {
            //This is just used to get access to the appsettings.json
            Configuration = new ConfigurationBuilder()
              .AddJsonFile("appsettings.json", optional: true, reloadOnChange: true)
              .AddEnvironmentVariables()
              .AddCommandLine(args)
              .Build();

            if (Configuration["OcpApimSubscriptionKey"].Equals("<PUT YOUR SUBSCRIPTION KEY HERE>"))
            {
                Console.WriteLine($"You need to update the appsettings.json to have your Client ID, Client Secret and Subscription Key");
                Console.WriteLine($"");
                Console.WriteLine($"Aborting");
            }
            else
            {

                Console.WriteLine($"Get token");
                await GetToken();
                Console.WriteLine($"Get accounts");
                await GetAccountList();
                Console.WriteLine($"Get transactions for all accounts");
                await GetTransactionsForAllAccounts();
            }
            Console.WriteLine("Hit ENTER to exit...");
            Console.ReadLine();
        }

        static async Task GetTransactionsForAllAccounts()
        {
            //Only Get a new token if the old one is expired.
            var token = AccessToken;

            //Represents the page parameter
            string page = "1";
            //Represents the pagesize parameter - note the default is 25 - so this is almost redundant
            string pageSize = "25";
            //Set the date range required
            string endTime = DateTime.Now.ToString("YYYY-MM-DD");
            string startTime = DateTime.Now.AddDays(-1).ToString("YYYY-MM-DD");
            var handler = new HttpClientHandler();
            handler.UseCookies = false;

            // In production code, don't destroy the HttpClient through using, but better reuse an existing instance
            // https://www.aspnetmonsters.com/2016/08/2016-08-27-httpclientwrong/
            using (var httpClient = new HttpClient(handler))
            {
                using (var request = new HttpRequestMessage(HttpMethod.Get, Configuration["apiEndPointBase"] + $"banking/accounts/transactions?page={page}&page-size={pageSize}&start-time={startTime}&end-time={endTime}"))
                {
                    //You SHOULD set a session-id, so calls can be traced from this systen all the way through the SISS APIs in case of any errors.  The SISS API will allocate one automatically
                    request.Headers.Add("session-id", Guid.NewGuid().ToString());
                    //You SHOULD set a x-fapi-interaction-id, so calls can be traced from this systen all the way through the SISS APIs in case of any errors.  The SISS API will allocate one automatically
                    request.Headers.Add("x-fapi-interaction-id", Guid.NewGuid().ToString());
                    //You MUST set a x-v to ensure you get the version of JSON returned that you need
                    request.Headers.Add("x-v", "1");
                    //You MUST supply your APIM Subscription Key
                    request.Headers.Add("Ocp-Apim-Subscription-Key", Configuration["OcpApimSubscriptionKey"]);

                    //You MUST supply the token collected from the Token Endpoint - In GetToken.  Note, you dont need to refresh this every call, as it lasts for 10 minutes.
                    httpClient.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("Bearer", token);
                    //The API only returns JSON, so good practice is to align with that.
                    httpClient.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json"));

                    //This gets the response
                    HttpResponseMessage response = await httpClient.SendAsync(request);

                    //Check it succeded - this throws an exception if it doesn't
                    response.EnsureSuccessStatusCode();

                    //Get the result as JSON
                    var jsonResponse = response?.Content?.ReadAsStringAsync();

                    //Write the JSON out
                    Console.WriteLine($"Result of call to get all Transactions: {(jsonResponse != null ? jsonResponse.Result : string.Empty)}");
                }
            }
        }

        static async Task GetAccountList()
        {
            //Only Get a new token if the old one is expired.
            var token = AccessToken;

            //Represents the page parameter
            string page = "1";
            //Represents the pagesize parameter - note the default is 25 - so this is almost redundant
            string pageSize = "25";
            //Represents the processing status parameter - A means return only accounts in an active state
            string processingStatus = "A";
            var handler = new HttpClientHandler();
            handler.UseCookies = false;

            // In production code, don't destroy the HttpClient through using, but better reuse an existing instance
            // https://www.aspnetmonsters.com/2016/08/2016-08-27-httpclientwrong/
            using (var httpClient = new HttpClient(handler))
            {
                using (var request = new HttpRequestMessage(HttpMethod.Get, Configuration["apiEndPointBase"] + $"banking/accounts?page={page}&page-size={pageSize}&processingStatus={processingStatus}"))
                {
                    //You SHOULD set a session-id, so calls can be traced from this systen all the way through the SISS APIs in case of any errors.  The SISS API will allocate one automatically
                    request.Headers.Add("session-id", Guid.NewGuid().ToString());
                    //You SHOULD set a x-fapi-interaction-id, so calls can be traced from this systen all the way through the SISS APIs in case of any errors.  The SISS API will allocate one automatically
                    request.Headers.Add("x-fapi-interaction-id", Guid.NewGuid().ToString());
                    //You MUST set a x-v to ensure you get the version of JSON returned that you need
                    request.Headers.Add("x-v", "1");
                    //You MUST supply your APIM Subscription Key
                    request.Headers.Add("Ocp-Apim-Subscription-Key", Configuration["OcpApimSubscriptionKey"]);

                    //You MUST supply the token collected from the Token Endpoint - In GetToken.  Note, you dont need to refresh this every call, as it lasts for 10 minutes.
                    httpClient.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("Bearer", token);
                    //The API only returns JSON, so good practice is to align with that.
                    httpClient.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json"));

                    //This gets the response
                    HttpResponseMessage response = await httpClient.SendAsync(request);

                    //Check it succeded - this throws an exception if it doesn't
                    response.EnsureSuccessStatusCode();

                    //Get the result as JSON
                    var jsonResponse = response?.Content?.ReadAsStringAsync();

                    //Write the JSON out
                    Console.WriteLine($"Result of call to get a List of Bank Accounts: {(jsonResponse != null ? jsonResponse.Result : string.Empty)}");
                }
            }

        }

        static async Task GetToken()
        {
            var handler = new HttpClientHandler();
            handler.UseCookies = false;

            // In production code, don't destroy the HttpClient through using, but better reuse an existing instance
            // https://www.aspnetmonsters.com/2016/08/2016-08-27-httpclientwrong/
            using (var httpClient = new HttpClient(handler))
            {
                using (var request = new HttpRequestMessage(HttpMethod.Post, Configuration["authEndPoint"]))
                {
                    request.Content = new FormUrlEncodedContent(new[]
                        {
                            new KeyValuePair<string, string>("grant_type", "client_credentials"),
                            new KeyValuePair<string, string>("client_id", Configuration["clientID"]),
                            new KeyValuePair<string, string>("client_secret", Configuration["clientSecret"]),
                            new KeyValuePair<string, string>("audience", Configuration["audience"]),
                    });
                    request.Content.Headers.ContentType = new MediaTypeHeaderValue("application/x-www-form-urlencoded");

                    HttpResponseMessage response = httpClient.SendAsync(request).Result;

                    response.EnsureSuccessStatusCode();

                    var responseBody = await response.Content.ReadAsStringAsync();

                    var accessToken = JObject.Parse(responseBody)["access_token"].ToString();

                    if (string.IsNullOrWhiteSpace(accessToken))
                        throw new ApplicationException($"Failed to get token from: {Configuration["authEndPoint"]} ");

                    AccessToken = accessToken;
                }
            }
        }
    }
}